/**********************************************************************
*<
FILE: 		    NET_TreeViewWrapper.ms

DESCRIPTION:    A wrapper for the .NET System.Windows.Forms.TreeView control

CREATED BY:     Chris P. Johnson
		
HISTORY:        Created April 18 2006

*>	Copyright Autodesdk (c) 2000, All Rights Reserved.
**********************************************************************/

struct TreeViewWraps
(
	m_dnColor = dotNetClass "System.Drawing.Color",
	m_iconClassType = dotnet.gettype "System.Drawing.Icon",
	m_bitmapClassType = dotnet.gettype "System.Drawing.Bitmap",
	m_bStyle    = dotNetClass "System.Windows.Forms.BorderStyle",

	function MXSColor_to_dotNetColor hColor = --New function
	(
		--[brief] Converts a maxscript color to a .NET color
		--[param] The maxscript color to convert
		--[return] A System.Drawing.Color object
		m_dnColor.fromARGB hColor.r hColor.g hColor.b
	),
	function IsIconFile stringFileName =
	(
		--[brief] Tells whether the filename is an icon or not.
		--[param] stringFileName - A maxscript string of the filename to query.
		--[return] true if the file is an icon (*.ico) file.
		local type = GetFileNameType stringFileName
		local result = false
		if type == ".ico" do ( result = true )
		result	
	),
	function InitImageList tv              \
					iconArray \
					pSize:0         \
					pTransparentColor:undefined =
	(
		--[brief] Add images to the treeNodes in a TreeView control. An imagelist
		--		is a structure that manages images for a control.
		--[param] tv - The System.Windows.Forms.TreeView class object
		--[param] iconArray - A maxscript array of string filenames for the images
		--		you want to add, System.Drawing.Icon instances, or System.Drawing.Bitmap instances
		--[param] pSize - The pixel dimension of the images you want to add. For instance passing
		--		in 16 will create 16x16 pixel images on your treenodes.
		--[param] pTransparentColor - The color to treat as transparent. The transparent
 		--		color is not rendered when the image is drawn
		--[return]The imagelist that was successfully created.
		--[Pre] 	Must pass the string file names as a symbolic pathname
		--     	i.e. "$maxSysIcons/mybitmap.bmp"
		--[Note] 	Icons and bitmaps are handled differently in .NET. This method
		--		only adds icons and bitmaps, to an imagelist.
		local hImgList = dotNetObject "System.Windows.Forms.ImageList"
		
		local image_size = 16 
		if pSize != 0 and (pSize > 2) then
			image_size = pSize 
		local imgSize 	= dotNetObject "System.Drawing.Size" image_size image_size
		hImgList.imagesize = imgSize

		if pTransparentColor != undefined and (classof pTransparentColor == color) then
		(	
			hImgList.transparentColor = m_dnColor.fromARGB pTransparentColor.r pTransparentColor.g pTransparentColor.b
		) 
		else
		(
			hImgList.transparentColor = m_dnColor.fromARGB 125 125 125	
		)
		
		for item in iconArray do
		(
			if ((classof item) == dotNetObject) then
			(
				local itemType = item.GetType()
				if (itemType == m_iconClassType or itemType == m_bitmapClassType) then
					hImgList.images.add item
				else
					throw "bad iconArray item type"
			)
			else
			(
				fName = (symbolicPaths.expandFileName item)
				if (IsIconFile fName ) then --Add icons
				(
					local icon  = dotNetObject "System.Drawing.Icon" fName
					hImgList.images.add icon
				)
				else --or Add bitmaps
				(
					local dnBitmap = dotNetObject "System.Drawing.Bitmap" fName
					hImgList.images.add dnBitmap
				)
			)
		)
		tv.ImageList = hImgList
	),
	---------------------------------------
	-- Misc Methods
	---------------------------------------
	function getHitNode arg tv =
	(
		--[brief] Finds the TreeNode that was hit, given a point
		--[param] arg - System.Windows.Forms.MouseEventArgs
		--[param] tv - The TreeView object that was hit
		--[return] Returns an object of the System.Windows.Forms.TreeNode class
		local position = dotNetObject "System.Drawing.Point" arg.x arg.y
		tv.GetNodeAt position
	),
	function ClearTvNodes tv =
	(
		--[brief] Deletes all treenodes in the treeview control.
		tv.nodes.clear()	
	),
	function CollapseAll tv =
	(
		--[brief] Collapses all treenodes in the treeview control.
		tv.collapseAll()
	),
	function ExpandAll tv =
	(
		--[brief] Expands all treenodes in the treeview control.
		tv.ExpandAll()	
	),
	function GetHitCoordinates tv =
	(
		--[brief] Use this method to get the hit coordinates on the treeview.
		--[param] tv - A System.Windows.Forms.TreeNode class object
		--[return] Returns the coordinates of the mouse as a .NET System.Drawing.Point object
		local screenHitPoint = tv.MousePosition
		tv.PointToClient screenHitPoint
	),
	---------------------------------------
	-- The Grand Initialization Method
	---------------------------------------
	function InitTreeView tv                 \
						pLabelEdit:    true  \
						pCheckBoxes:   false \ 
						pFullRowSel:   false \
						pAllowDrop:    false \
						pScrollable:   true  \
						pShowLines:    true  =
	(
		--[brief] Initialization Function to set general properties of a listview control
		------------------------------------
		-- user defined options
		------------------------------------
		-- Allow the user to edit item text.
		tv.LabelEdit     = pLabelEdit;
		-- Display check boxes
		tv.CheckBoxes    = pCheckBoxes;
		-- Select the item and subitems when selection is made.
		tv.FullRowSelect = pFullRowSel;
		-- Set Drag and Drop
		tv.allowDrop     = pAllowDrop
		-- Sort the items in the list in ascending order.
		tv.Sorted        = true
		-- Whether to display scroll bars.
		tv.scrollable    = pScrollable
		-- Show lines or not
		tv.showLines     = pShowLines
		
		tv.hideSelection 	= false
		
		tv.BorderStyle 	= m_bStyle.Fixed3D
		------------------------------------
		-- other options
		------------------------------------	
		-- Set Listview to System Colors
		local textColor   = MXSColor_to_dotNetColor (((colorman.getColor #text  )*255) as color)
		local windowColor = MXSColor_to_dotNetColor (((colorman.getColor #window)*255) as color)
			
		tv.backColor = windowColor
		tv.foreColor = textColor
	)	
)

global tvops = TreeViewWraps()